<?php
// src/Bot.php

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/AdvancedHttpClient.php';
require_once __DIR__ . '/DataHandler.php';
require_once __DIR__ . '/RateLimiter.php';
require_once __DIR__ . '/OpenAIHandler.php';
require_once __DIR__ . '/MessageProcessor.php';

class Bot {
    private $token;
    private $httpClient;
    private $dataHandler;
    private $rateLimiter;
    private $openAIHandler;
    private $messageProcessor;
    private $chatsFile;

    public function __construct() {
        $this->token = BOT_TOKEN;
        $this->httpClient = new AdvancedHttpClient();
        $this->dataHandler = new DataHandler();
        $this->rateLimiter = new RateLimiter();
        $this->openAIHandler = new OpenAIHandler();
        $this->messageProcessor = new MessageProcessor();
        $this->chatsFile = DATA_DIR . 'chats.json';
        
        // اطمینان از وجود فایل chats.json
        $this->ensureChatsFile();
    }

    private function ensureChatsFile() {
        if (!file_exists($this->chatsFile)) {
            file_put_contents($this->chatsFile, json_encode([]));
        }
    }

    public function handleUpdate($update) {
        if (isset($update['message'])) {
            // ذخیره اطلاعات چت قبل از پردازش
            $this->saveChatInfo($update['message']);
            $this->processMessage($update['message']);
        } elseif (isset($update['callback_query'])) {
            $this->processCallbackQuery($update['callback_query']);
        }
    }

    private function saveChatInfo($message) {
        try {
            $chat = $message['chat'];
            $from = $message['from'] ?? [];
            
            $chatId = $chat['id'];
            $chatType = $chat['type'];
            $firstName = $from['first_name'] ?? null;
            $lastName = $from['last_name'] ?? null;
            $username = $from['username'] ?? null;
            $title = $chat['title'] ?? null;

            $chats = [];
            if (file_exists($this->chatsFile)) {
                $content = file_get_contents($this->chatsFile);
                $chats = json_decode($content, true) ?: [];
            }

            $chatKey = (string)$chatId;
            $newEntry = [
                'id' => $chatId,
                'type' => $chatType,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'username' => $username,
                'title' => $title,
                'updated_at' => time()
            ];

            // ادغام با داده‌های موجود
            $existingEntry = $chats[$chatKey] ?? [];
            $chats[$chatKey] = array_merge($existingEntry, $newEntry);

            file_put_contents($this->chatsFile, json_encode($chats, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
            
        } catch (Exception $e) {
            error_log("Error saving chat info: " . $e->getMessage());
        }
    }

    private function processMessage($message) {
        $processed = $this->messageProcessor->process($message);

        switch ($processed['type']) {
            case 'ai_query':
                $this->handleAIQuery($processed['query'], $processed['chat_id'], $processed['user_id']);
                break;
            case 'command':
                $this->handleCommand($processed['command'], $processed['chat_id'], $processed['chat_type'], $processed['user_id']);
                break;
            case 'search_date':
                $this->handleSearchDate($processed['date'], $processed['chat_id']);
                break;
            case 'help':
                $this->sendHelpMessage($processed['chat_id']);
                break;
            case 'unknown_private':
                $this->sendHelpMessage($processed['chat_id']);
                break;
            case 'welcome':
                $this->sendWelcomeMessage($processed['chat_id'], $processed['message'], $processed['keyboard']);
                break;
            case 'ignored':
                // کاری نمی‌کنیم
                break;
        }
    }

    private function handleAIQuery($query, $chatId, $userId) {
        // 1. چک نرخ محدودیت فقط برای درخواست‌های معتبر AI
        $rateLimitCheck = $this->rateLimiter->isAllowed($userId);
        if ($rateLimitCheck !== true) {
            $this->sendMessage($chatId, $rateLimitCheck);
            return;
        }

        // 2. ارسال پیام در حال پردازش
        $this->sendMessage($chatId, "🤖 <b>سیف بات در حال پردازش سوال شما...</b>\n⏳ لطفاً کمی صبر کنید.");

        // 3. فراخوانی OpenAI
        $response = $this->openAIHandler->getResponse($query);

        // 4. ارسال پاسخ نهایی
        if ($response) {
            $this->sendMessage($chatId, $response);
        } else {
            $error_msg = "❌ متاسفم، در حال حاضر نمی‌توانم پاسخ شما را بدهم.\n";
            $error_msg .= "لطفاً:\n";
            $error_msg .= "• چند دقیقه دیگر دوباره امتحان کنید\n";
            $error_msg .= "• از منوی اصلی استفاده کنید\n";
            $error_msg .= "• با پشتیبانی تماس بگیرید: @" . SUPPORT_ID;
            $this->sendMessage($chatId, $error_msg);
            // ارسال گزارش به ادمین برای دیباگ
            $this->sendMessage(ADMIN_CHAT_ID, "⚠️ خطا در دریافت پاسخ از AI برای کاربر $chatId\nسوال: '$query'");
        }
    }

    private function handleCommand($command, $chatId, $chatType, $userId) {
        switch ($command) {
            case '/start':
                $this->showMainMenu($chatId);
                break;
            case '🤖 چت با هوش مصنوعی':
                $this->sendWelcomeMessage($chatId, $this->messageProcessor->getWelcomeMessage($chatType === 'private'));
                break;
            case '📜 قیمت‌های متنی':
            case '📜 متن نرخ':
            case 'متن نرخ':
                $this->sendTextPrices($chatId);
                break;
            case '🖼 آخرین تصویر':
                $this->sendLatestScreenshot($chatId);
                break;
            case '🔄 قیمت جدید':
                $this->sendNewScreenshot($chatId);
                break;
            case 'ℹ️ درباره ربات':
                $this->sendAbout($chatId);
                break;
            case '👨‍💼 پشتیبانی':
                $this->sendSupport($chatId);
                break;
            case '🛒 فروشگاه سیف گلد':
                $this->sendStore($chatId);
                break;
            case '🗂 آرشیو تصاویر':
                $this->sendArchive($chatId);
                break;
            case 'تصویر نرخ':
            case 'تصوير نرخ':
                $this->sendNewScreenshot($chatId); // ارسال تصویر جدید
                break;
            case '/search':
                $this->sendMessage($chatId, "📅 لطفاً یک تاریخ را به فرمت YYYY-MM-DD یا YYYY/MM/DD وارد کنید (مثلاً 2025-03-20 یا 1404/01/01):");
                break;
            case '/listchats':
                $this->handleListChats($chatId, $userId);
                break;
            default:
                $this->sendMessage($chatId, "دستور ناشناخته: $command");
        }
    }

    private function handleListChats($chatId, $userId) {
        // فقط کاربر با ID مشخص شده می‌تواند از این دستور استفاده کند
        if ($userId != 6685722743) {
            $this->sendMessage($chatId, "❌ شما دسترسی به این command را ندارید.");
            return;
        }

        $chats = [];
        if (file_exists($this->chatsFile)) {
            $content = file_get_contents($this->chatsFile);
            $chats = json_decode($content, true) ?: [];
        }

        if (empty($chats)) {
            $this->sendMessage($chatId, "❌ هیچ چتی ثبت نشده است.");
            return;
        }

        $message = "📊 <b>لیست چت‌های ثبت شده:</b>\n\n";
        $counter = 1;
        
        foreach ($chats as $chat) {
            $name = $chat['title'] ?? $chat['first_name'] ?? 'نامشخص';
            $username = $chat['username'] ? '@' . $chat['username'] : '---';
            
            $typeEmoji = '';
            $typeText = '';
            switch ($chat['type']) {
                case 'private': 
                    $typeEmoji = '👤';
                    $typeText = 'کاربر';
                    break;
                case 'group': 
                    $typeEmoji = '👥';
                    $typeText = 'گروه';
                    break;
                case 'supergroup': 
                    $typeEmoji = '👥👥';
                    $typeText = 'سوپرگروه';
                    break;
                case 'channel': 
                    $typeEmoji = '📢';
                    $typeText = 'کانال';
                    break;
                default: 
                    $typeEmoji = '❓';
                    $typeText = $chat['type'];
            }

            $message .= "<b>$counter-</b> $typeEmoji <code>" . $chat['id'] . "</code>\n";
            $message .= "  • نام: $name\n";
            $message .= "  • یوزرنیم: $username\n";
            $message .= "  • نوع: $typeText\n\n";
            $counter++;

            // جلوگیری از پیام خیلی طولانی
            if ($counter > 50) {
                $message .= "... (نمایش 50 مورد اول)";
                break;
            }
        }

        $message .= "📈 <b>جمع کل:</b> " . count($chats) . " چت ثبت شده";

        $this->sendMessage($chatId, $message);
    }

    private function handleSearchDate($date, $chatId) {
        $files = $this->dataHandler->getScreenshotArchive();
        $found = false;

        // نرمال‌سازی تاریخ ورودی
        $normalizedDate = str_replace(
            ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹', '/', '\\'],
            ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '-'],
            $date
        );

        foreach ($files as $file) {
            $basename = basename($file);
            $eng_basename = str_replace(
                ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹', '‏', '‌'],
                ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '', ''],
                $basename
            );

            // جستجو در نام فایل
            if (strpos($eng_basename, $normalizedDate) === 0) {
                $caption = "📸 تصویر قیمت مربوط به تاریخ: $date" . $this->dataHandler->getFileCreationDate($file);
                $this->sendPhoto($chatId, $file, $caption);
                $found = true;
                break;
            }
        }
        if (!$found) {
            $this->sendMessage($chatId, "❌ تصویری برای تاریخ $date یافت نشد.");
        }
    }

    private function sendTextPrices($chatId) {
        $data = $this->dataHandler->getLatestRates();
        if ($data) {
            $text = "📋 <b>آخرین قیمت‌ها:</b>\n\n";

            // قیمت طلا
            if (isset($data['gold']) && is_array($data['gold'])) {
                $text .= "<b>🪙 قیمت طلا:</b>\n";
                foreach ($data['gold'] as $goldItem) {
                    if (in_array($goldItem['name'], ['طلای 18 عیار', 'طلای 24 عیار', 'طلای آب‌شده نقدی'])) {
                        $text .= "• {$goldItem['name']}: <code>" . number_format($goldItem['price']) . " {$goldItem['unit']}</code> ({$goldItem['change_percent']}%)\n";
                    }
                }
                $text .= "\n";
            }

            // قیمت سکه
            if (isset($data['gold']) && is_array($data['gold'])) {
                $text .= "<b>🪙 قیمت سکه:</b>\n";
                $coinTypes = ['سکه یک گرمی', 'ربع سکه', 'نیم سکه', 'سکه امامی', 'سکه بهار آزادی'];
                foreach ($data['gold'] as $goldItem) {
                    if (in_array($goldItem['name'], $coinTypes)) {
                        $text .= "• {$goldItem['name']}: <code>" . number_format($goldItem['price']) . " {$goldItem['unit']}</code> ({$goldItem['change_percent']}%)\n";
                    }
                }
                $text .= "\n";
            }

            // قیمت ارز
            if (isset($data['currency']) && is_array($data['currency'])) {
                $text .= "<b>💱 قیمت ارز:</b>\n";
                $currencyTypes = ['دلار', 'یورو', 'درهم امارات', 'پوند', 'ین ژاپن', 'فرانک سوئیس'];
                foreach ($data['currency'] as $currencyItem) {
                    if (in_array($currencyItem['name'], $currencyTypes)) {
                        $text .= "• {$currencyItem['name']}: <code>" . number_format($currencyItem['price']) . " {$currencyItem['unit']}</code> ({$currencyItem['change_percent']}%)\n";
                    }
                }
                $text .= "\n";
            }

            // قیمت رمزارز
            if (isset($data['cryptocurrency']) && is_array($data['cryptocurrency'])) {
                $text .= "<b>🪙 قیمت رمزارز:</b>\n";
                $cryptoTypes = ['بیت‌کوین', 'اتریوم', 'تتر', 'ایکس‌آر‌پی', 'بی‌ان‌بی'];
                foreach ($data['cryptocurrency'] as $cryptoItem) {
                    if (in_array($cryptoItem['name'], $cryptoTypes)) {
                        $text .= "• {$cryptoItem['name']}: <code>{$cryptoItem['price']} {$cryptoItem['unit']}</code> ({$cryptoItem['change_percent']}%)\n";
                    }
                }
            }

            // --- اضافه کردن قیمت سکه‌های سیف ---
            $gold18kPrice = null;
            foreach ($data['gold'] as $item) {
                if ($item['name'] === 'طلای 18 عیار') {
                    $gold18kPrice = $item['price'];
                    break;
                }
            }

            if ($gold18kPrice) {
                $text .= "\n<b>🪙 قیمت سکه‌های سیف:</b>\n";
                $profit_margin = 0.15;
                $weight_1g = 1;
                $weight_halfg = 0.5;

                $price_1g = round($gold18kPrice * $weight_1g * (1 + $profit_margin));
                $price_halfg = round($gold18kPrice * $weight_halfg * (1 + $profit_margin));

                $text .= "• سکه 1 گرمی سیف: <code>" . number_format($price_1g) . " تومان</code>\n";
                $text .= "• سکه 0.5 گرمی سیف: <code>" . number_format($price_halfg) . " تومان</code>\n";
            } else {
                $text .= "\n<b>🪙 قیمت سکه‌های سیف:</b>\n";
                $text .= "• قیمت طلا برای محاسبه در دسترس نیست.\n";
            }

            $this->sendMessage($chatId, $text);

            // --- ارسال دکمه‌های خرید ---
            $buyButtons = $this->getBuyButtonsKeyboard();
            $this->sendMessage($chatId, "🛒 برای خرید سکه‌های سیف از دکمه‌های زیر استفاده کنید:", $buyButtons);

        } else {
            $this->sendMessage($chatId, "❌ متاسفانه نتوانستم قیمت‌ها را دریافت کنم.");
        }
    }

    private function sendLatestScreenshot($chatId) {
        $files = $this->dataHandler->getScreenshotArchive();
        if (!empty($files)) {
            $latest_file = $files[0];
            $caption = "📸 آخرین تصویر ذخیره‌شده" . $this->dataHandler->getFileCreationDate($latest_file);
            $this->sendPhoto($chatId, $latest_file, $caption, true);
        } else {
            $this->sendMessage($chatId, "❌ هیچ تصویری ذخیره نشده است.");
        }
    }

    private function sendNewScreenshot($chatId) {
        $file_name = $this->dataHandler->captureAndSaveScreenshot($this, $chatId);
        if ($file_name) {
             $caption = "📸 قیمت لحظه‌ای طلا" . $this->dataHandler->getFileCreationDate(UPLOAD_PATH . $file_name);
             $this->sendPhoto($chatId, UPLOAD_PATH . $file_name, $caption, true);
        }
    }

    private function sendAbout($chatId) {
        $about_text = "🪙 <b>سیف گلد</b>\n\n";
        $about_text .= "ما در سیف تلاش می کنیم تجربه ای متفاوت از خرید طلا را برای شما رقم بزنیم. از سکه های طلای کادویی با طراحی اختصاصی، بسته بندی های خاص و خدمات پس از فروش حرفه ای گرفته تا طلای آب شده برای سرمایه گذاری و زیورآلات طلای روزمره، همه چیز طوری طراحی و انتخاب شده که حس ارزشمند بودن را به شما و عزیزانتان منتقل کند.\n\n";
        $about_text .= "این ربات به شما امکان دسترسی سریع به قیمت‌های لحظه‌ای طلا و ارز را می‌دهد و همیشه در خدمت شماست.\n\n";
        $about_text .= "🌐 ما را در شبکه‌های اجتماعی دنبال کنید:";

        $about_keyboard = $this->getAboutKeyboard();
        $this->sendMessage($chatId, $about_text, $about_keyboard);
    }

    private function sendSupport($chatId) {
        $this->sendMessage($chatId, "📥 برای ارتباط با پشتیبانی از گزینه‌های زیر استفاده کنید:", $this->getSupportKeyboard());
    }

    private function sendStore($chatId) {
        $keyboard = ['inline_keyboard' => [[['text' => '🛍️ بازدید از فروشگاه', 'url' => 'https://safegold.store/']]]];
        $this->sendMessage($chatId, "🌐 برای بازدید از فروشگاه سیف گلد از دکمه زیر استفاده کنید:", $keyboard);
    }

    private function sendArchive($chatId) {
        $files = $this->dataHandler->getScreenshotArchive();
        if (!empty($files)) {
            $keyboard = $this->getArchiveKeyboard($files);
            $this->sendMessage($chatId, "📁 لطفاً یک تاریخ را از آرشیو انتخاب کنید:", $keyboard);
        } else {
            $this->sendMessage($chatId, "❌ هیچ تصویری در آرشیو وجود ندارد.");
        }
    }

    private function sendHelpMessage($chatId) {
        $help_message = "🔮 <b>دستورات ربات سیف گلد</b>\n\n";
        $help_message .= "📋 <b>منوی اصلی:</b>\n";
        $help_message .= "• از دکمه‌های پایین استفاده کنید\n\n";
        $help_message .= "🤖 <b>چت هوشمند:</b>\n";
        $help_message .= "• سوالات خود را با «" . BOT_PREFIX . "» شروع کنید\n\n";
        $help_message .= "📝 <b>مثال:</b>\n";
        $help_message .= "<code>" . BOT_PREFIX . " قیمت دلار چنده؟</code>\n\n";
        $help_message .= "💬 می‌تونید در مورد قیمت‌ها، محصولات و خدمات سوال کنید.";
        $keyboard = $this->messageProcessor->getMainKeyboard();
        $this->sendMessage($chatId, $help_message, $keyboard);
    }

    private function sendWelcomeMessage($chatId, $message, $keyboard = null) {
        $this->sendMessage($chatId, $message, $keyboard);
    }

    private function showMainMenu($chatId) {
        $keyboard = $this->messageProcessor->getMainKeyboard();
        $this->sendMessage($chatId, "🔮 <b>به ربات قیمت طلا و ارز سیف گلد خوش آمدید</b>\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:", $keyboard);
    }

    private function getAboutKeyboard() {
        return [
            'inline_keyboard' => [
                [
                    ['text' => '📺 آپارات', 'url' => 'https://www.aparat.com/safegoldstore'],
                    ['text' => '▶️ یوتیوب', 'url' => 'https://www.youtube.com/@safegoldstore']
                ],
                [
                    ['text' => '🐦 توییتر', 'url' => 'https://www.x.com/safegoldstore'],
                    ['text' => '📌 پینترست', 'url' => 'https://www.pinterest.com/safegoldstore']
                ],
                [
                    ['text' => '📸 اینستاگرام', 'url' => 'https://www.instagram.com/safegoldstore'],
                    ['text' => '🌐 وبسایت', 'url' => 'https://safegold.store/']
                ]
            ]
        ];
    }

    private function getSupportKeyboard() {
        return [
            'inline_keyboard' => [
                [
                    ['text' => '📞 تلگرام', 'url' => 'https://t.me/safegoldstore'],
                    ['text' => '📱 واتس‌اپ', 'url' => 'https://wa.me/+989364043825']
                ],
                [
                    ['text' => '☎️ تماس مستقیم', 'callback_data' => 'call_support']
                ]
            ]
        ];
    }

    private function getBuyButtonsKeyboard() {
        return [
            'inline_keyboard' => [
                [
                    ['text' => '🛒 خرید سکه 1 گرمی', 'url' => 'https://safegold.store/product/one-gram-safe-gold-coin/'],
                    ['text' => '🛒 خرید سکه 0.5 گرمی', 'url' => 'https://safegold.store/product/half-gram-safe-gold-coin/']
                ]
            ]
        ];
    }

    private function getArchiveKeyboard($files) {
        $keyboard = [];
        $row = [];
        $count = 0;
        $shownDates = [];

        foreach ($files as $file) {
            $basename = basename($file);
            $eng_basename = str_replace(
                ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹', '‏', '‌'],
                ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '', ''],
                $basename
            );

            $pattern = '/^(\d{4})-(\d{2})-(\d{2})_(\d{2})-(\d{2})\.jpg$/';
            if (preg_match($pattern, $eng_basename, $matches)) {
                $year = (int)$matches[1];
                $month = (int)$matches[2];
                $day = (int)$matches[3];

                $jdate = '';
                if ($year > 1500 && $year < 2100) {
                    $jdate_parts = DataHandler::gregorian_to_jalali($year, $month, $day);
                    if ($jdate_parts) {
                        $jyear = $jdate_parts[0];
                        $jmonth = $jdate_parts[1];
                        $jday = $jdate_parts[2];
                        $jdate = sprintf("%04d/%02d/%02d", $jyear, $jmonth, $jday);
                    } else {
                        $timestamp = filemtime($file);
                        $jdate = jdate('Y/m/d', $timestamp);
                    }
                } else {
                    $timestamp = filemtime($file);
                    $jdate = jdate('Y/m/d', $timestamp);
                }

                if ($jdate && !in_array($jdate, $shownDates)) {
                    $row[] = ['text' => $jdate, 'callback_data' => 'archive_' . $basename];
                    $count++;
                    $shownDates[] = $jdate;
                    if ($count % 2 === 0) {
                        $keyboard[] = $row;
                        $row = [];
                    }
                }
            } else {
                 $timestamp = filemtime($file);
                 $jdate = jdate('Y/m/d', $timestamp);
                 if ($jdate && !in_array($jdate, $shownDates)) {
                     $row[] = ['text' => $jdate, 'callback_data' => 'archive_' . $basename];
                     $count++;
                     $shownDates[] = $jdate;
                     if ($count % 2 === 0) {
                         $keyboard[] = $row;
                         $row = [];
                     }
                 }
            }
        }
        if (!empty($row)) {
            $keyboard[] = $row;
        }
        return ['inline_keyboard' => $keyboard];
    }

    public function sendMessage($chatId, $text, $replyMarkup = null) {
        if ($replyMarkup !== null && !is_array($replyMarkup)) {
            error_log("sendMessage: replyMarkup is not an array or null. It's: " . gettype($replyMarkup));
            $replyMarkup = null;
        }

        $url = "https://api.telegram.org/bot{$this->token}/sendMessage";
        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ];
        if ($replyMarkup) {
            $data['reply_markup'] = json_encode($replyMarkup);
        }

        $this->httpClient->post($url, json_encode($data), ['Content-Type: application/json']);
    }

    private function sendPhoto($chatId, $photoPath, $caption = '', $withDownloadButton = true) {
        if (!file_exists($photoPath)) {
            error_log("sendPhoto: File does not exist: $photoPath");
            return;
        }

        $url = "https://api.telegram.org/bot{$this->token}/sendPhoto";

        $postFields = [
            'chat_id' => $chatId,
            'photo' => new CURLFile($photoPath, 'image/jpeg', basename($photoPath)),
            'caption' => $caption
        ];

        if ($withDownloadButton) {
            $photoUrl = "https://safegold.store/telegram-bot/uploads/screenshots/" . basename($photoPath);
            $postFields['reply_markup'] = json_encode([
                'inline_keyboard' => [
                    [
                        ['text' => '📥 دانلود تصویر', 'url' => $photoUrl]
                    ]
                ]
            ]);
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postFields,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            error_log("cURL sendPhoto Error: " . $error);
        } elseif ($httpCode !== 200) {
            error_log("sendPhoto HTTP Error: " . $httpCode . " Response: " . $response);
        } else {
            error_log("sendPhoto successful to chat_id: $chatId, file: $photoPath");
        }
    }

    private function processCallbackQuery($callbackQuery) {
        $data = $callbackQuery['data'] ?? '';
        $chatId = $callbackQuery['message']['chat']['id'] ?? null;
        $messageId = $callbackQuery['message']['message_id'] ?? null;

        if ($chatId && $messageId && $data) {
            if (strpos($data, 'archive_') === 0) {
                $fileName = substr($data, 8);
                $fullPath = UPLOAD_PATH . $fileName;
                if (file_exists($fullPath)) {
                    $caption = "📸 تصویر انتخاب شده" . $this->dataHandler->getFileCreationDate($fullPath);
                    $this->sendPhoto($chatId, $fullPath, $caption, true);
                } else {
                    $this->sendMessage($chatId, "❌ فایل مورد نظر یافت نشد.");
                }
            } elseif ($data === 'call_support') {
                $this->sendMessage($chatId, "☎️ برای تماس مستقیم با پشتیبانی سیف گلد با شماره زیر تماس بگیرید:\n۰۲۱-۸۲۸۰۴۱۰۹");
            }
            
            // پاسخ دادن به کالبک کوئری
            $url = "https://api.telegram.org/bot{$this->token}/answerCallbackQuery";
            $data = [
                'callback_query_id' => $callbackQuery['id'],
                'text' => 'درخواست پردازش شد.',
                'show_alert' => false
            ];
            $this->httpClient->post($url, json_encode($data), ['Content-Type: application/json']);
        }
    }

    public function sendFallbackLinks($chatId) {
        $apis = getScreenshotAPIs();
        $apiKeys = array_keys($apis);

        $keyboard = [
            'inline_keyboard' => []
        ];

        for ($i = 0; $i < min(4, count($apiKeys)); $i++) {
            $apiName = $apiKeys[$i];
            $apiUrl = $apis[$apiName];
            $button = [
                'text' => "🔗 سرور $i",
                'url' => $apiUrl
            ];
            $keyboard['inline_keyboard'][] = [$button];
        }

        $message = "❌ دریافت خودکار تصویر با مشکل مواجه شد.\n\n";
        $message .= "با توجه به مسدود بودن سرور، می‌توانید به صورت دستی تصویر را از طریق لینک‌های زیر دریافت کنید:";

        $this->sendMessage($chatId, $message, $keyboard);
    }
}